import express from 'express';
import cors from 'cors';
import helmet from 'helmet';
import rateLimit from 'express-rate-limit';
import dotenv from 'dotenv';

// Import modules
import authRoutes from '@/modules/auth/routes';
import vendorRoutes from '@/modules/vendors/routes';
import subscriptionsRoutes from '@/modules/subscriptions/routes';
import aiRoutes from '@/modules/ai/router';
import remindersRoutes from '@/modules/reminders/router';
import shopRoutes from '@/modules/shoppinglists/router';
import customersRoutes from '@/modules/customers/router';
import invoicesRoutes from '@/modules/invoices/router';
import kycRoutes from '@/modules/kyc/router';
import greetingsRoutes from '@/modules/greetings/router';
import contributorsRoutes from '@/modules/contributors/router';
import referralsRoutes from '@/modules/referrals/router';
import notificationsRoutes from '@/modules/notifications/router';
import marketPricesRoutes from '@/modules/marketprices/router';
import sketchesRoutes from '@/modules/sketches/router';
import adminRoutes from '@/modules/admin/router';
import withdrawalsRoutes from '@/modules/withdrawals/router';
import consultationsRoutes from '@/modules/consultations/router';
import sellersRoutes from '@/modules/sellers/router';
import debugRoutes from '@/modules/debug/router';

// Import middleware
import { errorHandler, notFoundHandler } from '@/middlewares/errorHandler';

// Load environment variables
dotenv.config();

const app = express();
const PORT = process.env.PORT || 3000;

// Security middleware
app.use(helmet());
app.use(cors({
  origin: process.env.NODE_ENV === 'production' 
    ? ['https://your-frontend-domain.com'] 
    : ['http://localhost:3000', 'http://localhost:8080'],
  credentials: true
}));

// Rate limiting
const limiter = rateLimit({
  windowMs: parseInt(process.env.RATE_LIMIT_WINDOW_MS || '900000'), // 15 minutes
  max: parseInt(process.env.RATE_LIMIT_MAX_REQUESTS || '100'), // limit each IP to 100 requests per windowMs
  message: 'Too many requests from this IP, please try again later.'
});
app.use(limiter);

// Body parsing middleware
app.use(express.json({ limit: '10mb' }));
app.use(express.urlencoded({ extended: true, limit: '10mb' }));

// Health check endpoint
app.get('/health', (req, res) => {
  res.status(200).json({ 
    status: 'OK', 
    timestamp: new Date().toISOString(),
    environment: process.env.NODE_ENV 
  });
});

// API routes
app.use('/api/auth', authRoutes);
app.use('/api/vendors', vendorRoutes);
app.use('/api/subscriptions', subscriptionsRoutes);
app.use('/api/ai', aiRoutes);
app.use('/api/reminders', remindersRoutes);
app.use('/api/shop', shopRoutes);
app.use('/api/customers', customersRoutes);
app.use('/api/invoices', invoicesRoutes);
app.use('/api/kyc', kycRoutes);
app.use('/api/greetings', greetingsRoutes);
app.use('/api/contributors', contributorsRoutes);
app.use('/api/referrals', referralsRoutes);
app.use('/api/notifications', notificationsRoutes);
app.use('/api/market-prices', marketPricesRoutes);
app.use('/api/sketches', sketchesRoutes);
app.use('/api/admin', adminRoutes);
app.use('/api/withdrawals', withdrawalsRoutes);
app.use('/api/consultations', consultationsRoutes);
app.use('/api/sellers', sellersRoutes);
app.use('/api/debug', debugRoutes);

// Error handling middleware (must be last)
app.use(notFoundHandler);
app.use(errorHandler);

// Start server
app.listen(PORT, () => {
  console.log(`🚀 MODAVA Backend running on port ${PORT}`);
  console.log(`📊 Environment: ${process.env.NODE_ENV}`);
  console.log(`🔗 Health check: http://localhost:${PORT}/health`);
});

export default app; 